﻿using log4net;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.PersistenceService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.PersistenceService
{
    public class PersistenceService : IPersistenceService
    {
        private readonly ILog _logger;
        private readonly IPpmsContextHelper _ppmsContextHelper;
        private readonly IPpmsHelper _ppmsHelper;
        private PpmsContext _context = null;

        public PersistenceService(ILog logger, IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            _logger = logger;
            _ppmsContextHelper = ppmsContextHelper;
            _ppmsHelper = ppmsHelper;
        }

        public async Task<MapperResult> ModifyAccountsAsync(MapperResult result, string transactionId)
        {
            _logger.Info($"@@@@ INFO - Start ModifyAccountsAsync @@@@");

            if (result?.Details != null)
            {
                foreach (var item in result.Details)
                {
                    if (!item.IsValid) continue;

                    try
                    {
                        switch (item.Action)
                        {
                            case MapperAction.Insert:
                                await InsertEntity(item.Provider);
                                break;
                            case MapperAction.Update:
                                //await UpdateEntity(item.Provider);
                                await UpdateProvider(item.Provider);
                                break;
                            case MapperAction.Delete:
                                await DeleteEntity(item.Requests);
                                break;
                        }
                    }
                    catch (Exception ex)
                    {
                        _logger.Error($"@@@@ ERROR - Exception ModifyAccountsAsync @@@@", ex);
                        item.Provider = null;
                        item.ValidationMessage = $"An exception occurred committing provider information";
                    }
                }
            }

            _logger.Info($"@@@@ INFO - End ModifyAccountsAsync @@@@");
            return result;
        }

        public async Task DeleteAccountsAsync(MapperResult result, string transactionId)
        {
            _logger.Info($"@@@@ INFO - Start DeleteAccountsAsync @@@@");
            if (result?.Details != null)
            {
                await DeleteAccounts(result.Details);
            }
            _logger.Info($"@@@@ INFO - End DeleteAccountsAsync @@@@");
        }

        public async Task SaveBatchAsync(List<Validation> validationResults, DasMessage message)
        {
            var batchReferenceId = message.TransactionId;
            var networkId = await _ppmsHelper.GetNetworkIdByShorthand(message.SenderId);
            const int chunkSize = 50;

            if (validationResults != null && validationResults.Count > 0 && (!string.IsNullOrEmpty(batchReferenceId)) && (networkId.HasValue))
            {
                ResetContext();
                var context = await GetContext();

                var network = context.ppms_vaprovidernetworkSet.FirstOrDefault(x => x.Id == networkId.Value);
                if (network == null) throw new PpmsServiceException("Error writing Batch record for Schema Validation. Cannot find NetworkId for the given Provider");

                // Check for existing batch record
                var batch = await _ppmsHelper.GetBatch(message, false);
                SaveChangesResultCollection saveChanges;
                int batchStatus;
                if (batch == null)
                {
                    // batch does not exist, create
                    batch = new ppms_batch
                    {
                        ppms_transactionid = message.IsVaNetwork ? message.ConversationId : Guid.NewGuid().ToString(),
                        ppms_conversationid = message.ConversationId,
                        ppms_vaprovidernetwork_batch_network = network,
                        StatusCode = new OptionSetValue((int)ppms_batch_StatusCode.DataReceived)
                    };
                    context.AddObject(batch);

                    batchStatus = batch.StatusCode.Value;

                    // create batch record, if it doesn't exist
                    saveChanges = context.SaveChanges();
                    CheckSaveChanges(saveChanges);
                    batch = null;
                }
                else
                {
                    _logger.Info($"@@@@ DEBUG - Batch found @@@@");
                    batchStatus = batch.StatusCode.Value;
                }

                _logger.Info($"@@@@ INFO - Processing batch details @@@@");
                var size = validationResults.Count;
                var i = 0;
                bool processingRequired = false;
                var timer = new Stopwatch();
                timer.Start();

                // process validation results, create batch detail records
                foreach (var validationResult in validationResults)
                {
                    try
                    {
                        i++;

                        // add batch to current context
                        if (batch == null)
                        {
                            batch = await _ppmsHelper.GetBatch(message, false);
                            context.Attach(batch);
                        }

                        if (batch == null)
                        {
                            _logger.Error($"@@@@ ERROR - Unable to establish Batch Records @@@@");
                            return;
                        }

                        var transactionType = GetTransactionType(validationResult.TransactionType);
                        // If invalid result of deactivation, validation not required
                        var detailValidationComplete = !validationResult.IsValid || validationResult.TransactionType != 0;

                        // If one record is valid, then need to process on the PPMS side
                        processingRequired = processingRequired || !detailValidationComplete;

                        var batchDetail = new ppms_batchdetail
                        {
                            ppms_name = validationResult.ProviderName,
                            ppms_isvalid = validationResult.IsValid,
                            ppms_providerid = validationResult.ProviderId,
                            ppms_transactiontype = new OptionSetValue(transactionType),
                            ppms_provider = string.IsNullOrEmpty(validationResult.CorrelationId) ? null : new EntityReference(Account.EntityLogicalName, new Guid(validationResult.CorrelationId))
                        };
                        if (detailValidationComplete) batchDetail.ppms_validationcompletedate = DateTime.Now;
                        if (!context.IsAttached(batch)) context.Attach(batch);
                        context.AddRelatedObject(batch, new Relationship("ppms_batch_batchdetail_batch"), batchDetail);

                        if (!validationResult.IsValid)
                        {
                            var batchDetailResult = new ppms_batchdetailresult
                            {
                                ppms_isvalid = validationResult.IsValid,
                                ppms_name = validationResult.ProviderId,
                                ppms_entitytype = validationResult.ErrorSource,
                                ppms_result = validationResult.IsValid
                                    ? "Your provider file record has been successfully verified and updated in the Provider Profile Management System (PPMS)."
                                    : "Your provider file record is invalid. Please review the information as indicated and resubmit the provider once the information has been corrected.",
                                ppms_message = validationResult.Result,
                            };
                            context.AddRelatedObject(batchDetail, new Relationship("ppms_batchdetail_batchdetailresult"), batchDetailResult);
                        }

                        if (i % chunkSize == 0 || i == size)
                        {
                            _logger.Info($"Saving batch detail {i} of {size} [{timer.Elapsed}]");
                            saveChanges = context.SaveChanges();
                            CheckSaveChanges(saveChanges);
                            batch = null;
                            // Re-establish context
                            if (i % (chunkSize * 10) == 0)
                            {
                                ResetContext();
                                context = await GetContext();
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        _logger.Error($"@@@@ ERROR - There was a problem Saving Batch Records @@@@", ex);
                    }
                }

                _logger.Info($"Batch save complete. [{timer.Elapsed}]");
                timer.Stop();

                // change batch status once processing is complete
                if (batchStatus == (int)ppms_batch_StatusCode.DataReceived && message.Index == DasMessage.FinalFileIndicator)
                {
                    await _ppmsHelper.UpdateBatch(message, "Provider import process complete", processingRequired ? (int)ppms_batch_StatusCode.Processing : (int)ppms_batch_StatusCode.ValidationComplete);
                }
            }
            else if (!networkId.HasValue)
            {
                _logger.Error($"@@@@ ERROR - There was a problem Saving Batch Records. @@@@");
            }
        }

        private static void CheckSaveChanges(SaveChangesResultCollection saveChanges)
        {
            if (!saveChanges.HasError) return;

            var sb = new StringBuilder();
            foreach (var saveChange in saveChanges)
            {
                if (saveChange.Error != null)
                {
                    sb.AppendFormat("Error: {0}\n", saveChange.Error.Message);
                }
            }

            throw new PpmsServiceException($"Failure occurred while saving Batch records: {sb}");
        }

        private static int GetTransactionType(int transactionType)
        {
            switch (transactionType)
            {
                case (int)TransactionTypeList.Update: return (int)ppms_ServiceTransactionType.Update;
                case (int)TransactionTypeList.DeactivateProvider: return (int)ppms_ServiceTransactionType.DeactivateProvider;
                case (int)TransactionTypeList.DeactivateRelationship: return (int)ppms_ServiceTransactionType.DeactivateRelationship;
            }

            throw new PpmsServiceException("Error writing Batch record for Schema Validation. Invalid TransactionType.");
        }

        private async Task InsertEntity(Entity entity)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                if (!context.IsAttached(entity))
                {
                    context.Attach(entity);
                }

                context.SaveChanges();
            }
        }

        private async Task UpdateProvider(Account provider)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                if (!context.IsAttached(provider))
                {
                    context.Attach(provider);
                }

                context.SaveChanges();
            }
        }

        private async Task DeleteEntity(IList<SetStateRequest> requests)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceContext())
            {
                foreach (var request in requests)
                {
                    proxy.Execute(request);
                }
            }
        }

        private async Task DeleteAccounts(IList<MapperResultDetail> accounts)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceContext())
            {
                foreach (var account in accounts)
                {
                    foreach (var request in account.Requests)
                    {
                        proxy.Execute(request);
                    }
                }
            }
        }

        #region Context

        private async Task<PpmsContext> GetContext()
        {
            if (_context == null)
                _context = await _ppmsContextHelper.GetContextAsync();
            return _context;
        }

        private void ResetContext()
        {
            _context = null;
        }

        #endregion
    }
}